// stdlib
#include <iostream>
#include <string>

// STL
#include <vector>

// boost
#include <boost/foreach.hpp>
#include <boost/unordered_map.hpp>

// ROOT
#include <TFile.h>
#include <TTree.h>
#include <TKey.h>
#include <TList.h>
#include <TObjArray.h>

// Reflex
#include <Reflex/Type.h>

// Cintex
#include <Cintex/Cintex.h>

// local
#include <Event/Track.h>
#include <Solar/DataVector.h>

using namespace std;
using boost::unordered_map;
using Reflex::Type;

using Solar::DataObject;
using Solar::DataVector;
using Event::Track;

int main() {

   ROOT::Cintex::Cintex::Enable();

   TFile file( "test.root", "read" );

   TList* keys = file.GetListOfKeys();
   TIter listIter( keys );

   typedef unordered_map< string, TTree* > treeMap_t;
   treeMap_t trees;
   
   while ( TKey* key = static_cast< TKey* >( listIter.Next() ) ) {
      string className( key->GetClassName() );
      if (  className == "TTree" ) {
         string treeName( key->GetName() );
         TTree* tree = static_cast< TTree* >( file.Get( treeName.c_str() ) );
         trees.insert( make_pair( treeName, tree ) );
      }
   }

   typedef unordered_map< string, DataObject* > dataMap_t;
   dataMap_t dataMap;

   BOOST_FOREACH( const treeMap_t::value_type& entry, trees ) {
      string treeName = entry.first;
      TTree* tree = entry.second;
      TObjArray* branches = tree->GetListOfBranches();
      if ( branches->GetEntries() != 1 ) {
         cerr << "Error, number of branches in the tree is not 1." << endl;
         throw std::exception();
      }
      TBranch* branch = static_cast< TBranch* >( branches->At( 0 ) );
      const char* branchName = branch->GetName();
      dataMap.insert( make_pair( treeName, static_cast< DataObject* >( 0 ) ) );
      tree->SetBranchAddress( branchName, &( dataMap[ treeName ] ) );
   }

   DataVector< Track >* v = 0;
   
   treeMap_t::const_iterator it_tree = trees.begin();
   unsigned long entries = it_tree->second->GetEntries();
   
   for ( unsigned long i = 0; i < entries; i++ ) {
      BOOST_FOREACH( const treeMap_t::value_type& entry, trees ) {
         entry.second->GetEntry( i );
      }
      BOOST_FOREACH( const dataMap_t::value_type& entry, dataMap ) {
         v = dynamic_cast< DataVector< Track >* >( entry.second );
         if ( !v ) continue;
         BOOST_FOREACH( Track* track, *v ) {
            cout << track->charge() << " " << track->momentum() << endl;
         }
      }
   }
   
   return 0;
}
