// This file is part of Solar, a small framework for data processing.
// Copyright (C) 2009 Roel Aaij

// miniTPC is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.

// miniTPC is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef DATAVECTOR_H
#define DATAVECTOR_H 1

// Include files
#include <iterator>
#include <string>
#include <sstream>
#include <exception>
#include <stdexcept>
#include <iostream>
#include <typeinfo>

// STL
#include <map>

// boost
#include <boost/noncopyable.hpp>

// Solar
#include <Solar/IContainer.h>
#include <Solar/DataObject.h>

// Reflex
// #include <Reflex/Type.h>

/** @class Solar::DataVector DataVector.h
 *  A templated vector like class that is a DataObject.
 *  This class can be stored in the data store.
 *
 *  @author Roel Aaij
 *  @date   2008-08-22
 */

namespace Solar {

   template <class T, class Allocator = std::allocator< T* > >
   class DataVector : public IContainer, public DataObject,
                      public boost::noncopyable, private Allocator {

   private:

      // data members
      typename Allocator::pointer m_first;
      typename Allocator::pointer m_last;
      typename Allocator::size_type m_size;
      typename Allocator::size_type m_capacity;

      typedef std::map< const T*, unsigned int > addressMap_t;
      addressMap_t m_addresses;

   public:

      typedef T* value_type;
      typedef typename Allocator::size_type size_type;

      typedef typename Allocator::reference reference;
      typedef typename Allocator::const_reference const_reference;

      typedef typename Allocator::difference_type difference_type;

      typedef typename Allocator::pointer pointer;
      typedef typename Allocator::const_pointer const_pointer;

      typedef Allocator allocator_type;

      allocator_type GetAllocator() const {
         return static_cast<const Allocator&>(*this);
      }

      /**
       * Constuctor
       *
       * @param size the initial size of the DataVector
       */
      explicit DataVector( size_type n = 0, T* t = 0, const Allocator& a = Allocator() )
         : Allocator( a ), m_first( 0 ), m_last( 0 ), m_size( n ),
           m_capacity( 10 )
      {
         if ( n > m_capacity ) {
            m_capacity = n + n / 2;
         }

         m_first = Allocator::allocate( m_capacity );
         if ( m_size != 0 ) {
            size_type i;
            try {
               for ( i = 0; i < m_size; ++i ) {
                  Allocator::construct( m_first + i, t );
                  if ( t != 0 ) m_addresses.insert( std::make_pair( t, i ) );
               }
            }
            catch( ... ) {
               for( size_type j = 0; j < i; ++j ) {
                  Allocator::destroy( m_first + j );
                  const T* tmp = *( m_first + j );
                  if ( tmp ) m_addresses.erase( tmp  );
               }
               Allocator::deallocate( m_first, n );
               throw;
            }
         }
         m_last = m_first + m_size;
      }

      /**
       * Destructor
       */
      virtual ~DataVector()
      {
         if ( m_first ) {
            for (size_type i = 0; i < m_size; ++i) {
               Allocator::destroy( m_first + i );
               T* tmp = *( m_first + i );
               if ( tmp ) {
                  m_addresses.erase( tmp );
                  tmp->SetContainer( 0 );
                  delete tmp;
               }
            }
            Allocator::deallocate( m_first, m_capacity );
         }
      }

      /**
       * @class mTPC::DataVector::iterator
       * A member class that describes an iterator for use with
       * DataVector.
       */
      class iterator :
         public std::iterator< std::random_access_iterator_tag, T* > {
      private:

         pointer m_element;

      public:

         /**
          * Constuctor
          *
          * @param element the element this iterator will point to
          */
         iterator( pointer element )
            : m_element( element )
         {
         }

         /**
          * Copy constructor
          *
          * @param other the other iterator
          */
         iterator( const iterator& other )
            : m_element(other.m_element)
         {
         }

         /**
          * Destructor
          */
         ~iterator() {}

         /**
          * Operator +
          *
          * @param nplaces the  number of steps to increment the iterator
          *
          * @return a reference to the iterator
          */
         iterator operator+(int nplaces)
         {
            iterator tmp(*this);
            tmp += nplaces;
            return tmp;
         }

         /**
          * Operator +=
          *
          * @param nplaces the  number of steps to increment the iterator
          *
          * @return a reference to the iterator
          */
         iterator& operator+=(int nplaces)
         {
            m_element += nplaces;
            return *this;
         }

         /**
          * Pre-increment operator
          *
          * @return a reference to the iterator
          */
         iterator& operator++()
         {
            ++m_element;
            return *this;
         }

         /**
          * Post-increment operator
          *
          * @param dummy
          *
          * @return an iterator that points to the element before incementation.
          */
         iterator operator++(int dummy)
         {
            iterator tmp(*this);
            ++(*this);
            return tmp;
         }

         /**
          * Operator -
          *
          * @param nplaces the number of steps to decrement the iterator
          *
          * @return a reference to the iterator
          */
         iterator operator-(int nplaces)
         {
            iterator tmp(*this);
            tmp -= nplaces;
            return tmp;
         }

         /**
          * Operator -
          *
          * @param the other iterator to calculate the distance to
          *
          * @return the difference
          */
         difference_type operator-(const iterator& other)
         {
            return m_element - other.m_element;
         }

         /**
          * Operator -=
          *
          * @param nplaces the number of steps to decrement the iterator
          *
          * @return a reference to the iterator
          */
         iterator& operator-=(int nplaces)
         {
            m_element -= nplaces;
            return *this;
         }

         /**
          * Post decrement operator
          *
          * @param dummy
          *
          * @return a iterator that points to the element
          * before decrementation.
          */
         iterator operator--(int dummy)
         {
            iterator tmp(*this);
            --(*this);
            return tmp;
         }

         /**
          * Pre decrement operator
          *
          * @return a reference to the iterator.
          */
         iterator& operator--()
         {
            --m_element;
            return *this;
         }

         /**
          * Operator =
          *
          * @param other the other iterator
          *
          * @return a reference to the iterator
          */
         iterator& operator=(const iterator& other)
         {
            if (&other != this) {
               m_element = other.m_element;
            }
            return *this;
         }

         /**
          * Operator ==
          *
          * @param other the other iterator
          *
          * @return true if the iterator is equal the the other iterator
          */
         bool operator==(const iterator& other)
         {
            return m_element == other.m_element;
         }

         /**
          * operator !=
          *
          * @param other the other iterator
          *
          * @return true if the iterator is not equal to the other iterator
          */
         bool operator!=(const iterator& other)
         {
            return m_element != other.m_element;
         }

         /**
          * Dereference operator
          *
          * @return a reference to the element pointed to.
          */
         reference operator*() const
         {
            return *m_element;
         }

         /**
          * Dereference operator
          *
          *
          * @return a pointer to the element pointed to.
          */
         pointer operator->() const
         {
            typename DataVector<T>::iterator tmp = *this;
            T& val = *tmp;
            return (&val);
         }

         /**
          * Random access operator.
          * the iterator will point to the element at index
          *
          * @param index the index
          *
          * @return reference to the element now pointed to
          */
         iterator& operator[](size_type index) const
         {
            m_element += index;
            return *this;
         }

         /**
          * Smaller than operator
          *
          * @param other the other iterator
          *
          * @return true if this iterator is smaller than other.
          */
         bool operator<(const iterator& other) const
         {
            return m_element < other.m_element;
         }

         /**
          * Larger than operator
          *
          * @param other the other iterator
          *
          * @return true if this iterator is larger than other.
          */
         bool operator>(const iterator& other) const
         {
            return m_element > other.m_element;
         }
      };

      /**
       * @class mTPC::DataVector::const_iterator
       * A member class that describes a const iterator for use with
       * DataVector.
       */
      class const_iterator :
         public std::iterator< std::random_access_iterator_tag, T* > {
      private:

         pointer m_element;

      public:

         /**
          * Constuctor
          *
          * @param element the element this iterator will point to
          */
         const_iterator(pointer element)
            : m_element(element)
         {
         }

         /**
          * Copy constructor
          *
          * @param other the other iterator
          */
         const_iterator(const const_iterator& other)
            : m_element(other.m_element)
         {
         }

         /**
          * Destructor
          */
         ~const_iterator() {}

         /**
          * Operator +
          *
          * @param nplaces the  number of steps to increment the iterator
          *
          * @return a reference to the iterator
          */
         const_iterator operator+(int nplaces)
         {
            iterator tmp(*this);
            tmp += nplaces;
            return tmp;
         }

         /**
          * Operator +=
          *
          * @param nplaces the  number of steps to increment the iterator
          *
          * @return a reference to the iterator
          */
         const_iterator& operator+=(int nplaces)
         {
            m_element += nplaces;
            return *this;
         }

         /**
          * Pre-increment operator
          *
          * @return a reference to the iterator
          */
         const_iterator& operator++()
         {
            ++m_element;
            return *this;
         }

         /**
          * Post-increment operator
          *
          * @param dummy
          *
          * @return an iterator that points to the element
          * before incementation.
          */
         const_iterator operator++(int dummy)
         {
            const_iterator tmp(*this);
            ++(*this);
            return tmp;
         }

         /**
          * Operator -
          *
          * @param nplaces the number of steps to decrement the iterator
          *
          * @return a reference to the iterator
          */
         const_iterator operator-(int nplaces)
         {
            const_iterator tmp(*this);
            tmp -= nplaces;
            return tmp;
         }

         /**
          * Operator -
          *
          * @param the other iterator to calculate the distance to
          *
          * @return the difference
          */
         difference_type operator-(const const_iterator& other)
         {
            return m_element - other.m_element;
         }

         /**
          * Operator -=
          *
          * @param nplaces the number of steps to decrement the iterator
          *
          * @return a reference to the iterator
          */
         const_iterator& operator-=(int nplaces)
         {
            m_element -= nplaces;
            return *this;
         }

         /**
          * Post decrement operator
          *
          * @param dummy
          *
          * @return a iterator that points to the element
          * before decrementation.
          */
         const_iterator operator--(int dummy)
         {
            const_iterator tmp(*this);
            --(*this);
            return tmp;
         }

         /**
          * Pre decrement operator
          *
          * @return a reference to the iterator.
          */
         const_iterator& operator--()
         {
            --m_element;
            return *this;
         }

         /**
          * Operator =
          *
          * @param other the other iterator
          *
          * @return a reference to the iterator
          */
         const_iterator& operator=(const const_iterator& other)
         {
            if (&other != this) {
               m_element = other.m_element;
            }
            return *this;
         }

         /**
          * Operator ==
          *
          * @param other the other iterator
          *
          * @return true if the iterator is equal the the other iterator
          */
         bool operator==(const const_iterator& other)
         {
            return m_element == other.m_element;
         }


         /**
          * operator !=
          *
          * @param other the other iterator
          *
          * @return true if the iterator is not equal to the other iterator
          */
         bool operator!=(const const_iterator& other)
         {
            return m_element != other.m_element;
         }

         /**
          * Dereference operator
          *
          * @return a reference to the element pointed to.
          */
         const_reference operator*() const
         {
            return *m_element;
         }

         /**
          * Dereference operator
          *
          *
          * @return a pointer to the element pointed to.
          */
         const_pointer operator->() const
         {
            typename DataVector<T>::const_iterator tmp = *this;
            const T& val = *tmp;
            return (&val);
         }

         /**
          * Random access operator.
          * the iterator will point to the element at index
          *
          * @param index the index
          *
          * @return reference to the element now pointed to
          */
         const_iterator& operator[](unsigned int index) const
         {
            m_element += index;
            return *this;
         }

         /**
          * Smaller than operator
          *
          * @param other the other const_iterator
          *
          * @return true if this iterator is smaller than other.
          */
         bool operator<(const const_iterator& other) const
         {
            return m_element < other.m_element;
         }

         /**
          * Larger than operator
          *
          * @param other the other const_iterator
          *
          * @return true if this iterator is larger than other.
          */
         bool operator>(const const_iterator& other) const
         {
            return m_element > other.m_element;
         }
      };

      /**
       * Get an iterator that points to the beginning of this DataVector.
       *
       * @return the iterator.
       */
      iterator begin()
      {
         return iterator( m_first );
      }

      /**
       * Get an iterator that points to the beginning of this DataVector.
       * const version.
       *
       * @return the iterator.
       */
      const_iterator begin() const
      {
         return const_iterator( m_first );
      }

      /**
       * Get an iterator that points to 1 item past the end of this DataVector.
       *
       * @return the iterator.
       */
      iterator end()
      {
         return iterator( m_last );
      }

      /**
       * Get an iterator that points to 1 item past the end of this DataVector.
       * const version.
       *
       * @return the iterator.
       */
      const_iterator end() const
      {
         return const_iterator( m_last );
      }

      /**
       * Get the size of the DataVector.
       *
       * @return the size of the DataVector
       */
      unsigned int size() const
      {
         return m_size;
      }

      /**
       * Is the DataVector empty?
       *
       * @return empty?
       */
      bool empty() const
      {
         return !m_size;
      }

      /**
       * resize the DataVector.
       *
       * @param size the new size.
       */
      void resize( size_type size )
      {
         if ( size == m_size ) return;

         size_type capacity = 0;
         size_type nCopy = 0;
         if (size <= m_capacity) {
            if ( size <= m_size ) {
               nCopy = size;
            } else {
               nCopy = m_size;
            }
            if (size < 6) {
               capacity = 10;
            } else if (size < 2 * m_capacity / 3) {
               capacity = 2 * m_capacity / 3;
               if (capacity < 10) capacity = 10;
            } else {
               capacity = m_capacity;
            }
         } else {
            nCopy = m_size;
            capacity = size + size / 2;
         }

         std::cout << "m_capacity = " << m_capacity
                   << ", capacity =  " << capacity
                   << ", m_size =  " << m_size
                   << ", size =  " << size
                   << ", nCopy = " << nCopy << std::endl;

         pointer first = Allocator::allocate( capacity );
         size_type i;
         try {
            // Copy existing items
            for ( i = 0; i < nCopy; ++i ) {
               Allocator::construct( first + i, *( m_first + i ) );
            }
            // Create new ones.
            for ( i = nCopy; i < size; ++i ) {
               Allocator::construct( first + i, 0 );
            }
         }
         catch(...) {
            for( size_type j = 0; j < i; ++j ) {
               Allocator::destroy( first + j );
               const T* tmp = *( m_first + j );
               if ( tmp ) m_addresses.erase( tmp );
            }
            Allocator::deallocate( first, capacity );
            throw;
         }
         // delete old stuff.
         for( size_type i = 0; i < m_size; ++i ) {
            if ( i >= size ) {
               T* tmp = *( m_first + i );
               if ( tmp ) {
                  tmp->SetContainer( 0 );
                  m_addresses.erase( tmp );
                  delete tmp;
               }
            }
            Allocator::destroy( m_first + i );
         }
         Allocator::deallocate( m_first, m_capacity );
         m_first = first;
         m_capacity = capacity;
         m_size = size;
         m_last = m_first + m_size;
      }

      /**
       * Get the capacity of the DataVector
       *
       * @return the capacity
       */
      size_type capacity() const
      {
         return m_capacity;
      }

      /**
       * Get data object
       * const version
       *
       * @param index index of object
       *
       * @return const reference to the object
       */
      const_reference at( unsigned int index ) const
      {
         std::stringstream msg;
         msg << "index " << index << " out of range";
         if ( index >= m_size ) throw std::out_of_range( msg.str() );
         return *( m_first + index );
      }

      /**
       * Get data object
       *
       * @param index index of object
       *
       * @return const reference to the object
       */
      reference at( unsigned int index )
      {
         std::stringstream msg;
         msg << "index " << index << " out of range";
         if ( index >= m_size ) throw std::out_of_range( msg.str() );
         return *( m_first + index );
      }

      /**
       * Add object to the head of the DataVector
       *
       * @param data the object to add
       */
      void push_back( const_reference data )
      {
         size_type size = m_size + 1;
         if ( size <= m_capacity ) {
            try {
               Allocator::construct( m_first + m_size, data );
               if ( data ) m_addresses.insert( std::make_pair( data, m_size ) );
            }
            catch(...) {
               Allocator::destroy( m_first + m_size );
               if ( data ) m_addresses.erase( data );
               throw;
            }
            m_size = size;
            ++m_last;
         } else {
            resize( m_size + 1 );
            Allocator::destroy( m_first + m_size - 1 );
            const T* tmp = *( m_first + m_size - 1 );
            if ( tmp ) m_addresses.erase( tmp );
            try {
               Allocator::construct( m_first + m_size - 1, data );
               if ( data ) m_addresses.insert( std::make_pair( data, m_size - 1 ) );
            }
            catch(...) {
               Allocator::destroy(m_first + m_size - 1);
               const T* tmp = *( m_first + m_size - 1 );
               if ( tmp ) m_addresses.erase( tmp );
               throw;
            }
         }
         data->Release();
         data->SetContainer( this );
      }

      /**
       * Get and remove the data object at the head of the DataVector.
       *
       * @return the object
       */
      void pop_back()
      {
         resize( m_size - 1 );
      }

      void Remove( size_type index )
      {
         typename DataVector<T>::iterator it( m_first + index );
         erase( it );
      }

      /**
       * Get the size of the DataVector.
       *
       * @return the size of the DataVector
       */
      unsigned int Size() const
      {
         return m_size;
      }

      /**
       * Is the DataVector empty?
       *
       * @return empty?
       */
      bool Empty() const
      {
         return !m_size;
      }

      /**
       * Remove object from the DataVector.
       *
       * @param first iterator that points to the object that is to be removed.
       */
      void erase( const iterator& iter )
      {
         typename DataVector<T>::iterator end( iter );
         erase( iter, ++end );
      }

      /**
       * Remove all object between first and last from the DataVector.
       * last is not removed
       * @param first iterator that points to the first object to be removed.
       * @param last iterator that points one beyond the last object to be
       * removed.
       */
      void erase( const iterator& left, const iterator& right )
      {
         size_type capacity = 0;
         size_type nCopyBottom = 0;
         size_type nCopyTop = 0;
         size_type size = distance( left, right );
         nCopyBottom = distance( begin(), left );
         nCopyTop = distance( right, end() );
         if (size < 6) {
            capacity = 10;
         } else if (size < 2 * m_capacity / 3) {
            capacity = 2 * m_capacity / 3;
            if (capacity < 10) capacity = 10;
         } else {
            capacity = m_capacity;
         }

         pointer first = Allocator::allocate( capacity );
         size_type i;
         try {
            // Copy Bottom half
            for (i = 0; i < nCopyBottom; ++i) {
               Allocator::construct( first + i, *( m_first + i ) );
            }
            // Copy the top half
            for (i = 0; i < nCopyTop; ++i) {
               Allocator::construct( first + nCopyBottom + i, *( &*right + i ) );
            }
         }
         catch(...) {
            for(size_type j = 0; j < i; ++j) {
               Allocator::destroy( first + j );
            }
            Allocator::deallocate( first, capacity );
            throw;
         }
         // delete old stuff.
         for ( typename DataVector< T >::iterator it = left; it != right; ++it ) {
            T* tmp = *it;
            if ( tmp ) {
               m_addresses.erase( tmp );
               tmp->SetContainer( 0 );
               delete tmp;
            }
         }

         for(size_type i = 0; i < m_size; ++i) {
            Allocator::destroy( m_first + i );
         }
         Allocator::deallocate(m_first, m_capacity);

         m_first = first;
         m_capacity = capacity;
         m_size = size;
         m_last = m_first + m_size;
      }

      /**
       * Remove all objects from the DataVector.
       *
       */
      void clear()
      {
         resize( 0 );
      }

      /**
       * Random member access operator.
       * const version
       *
       * @param index the index of the object.
       *
       * @return const reference the the object.
       */
      const_reference operator[]( unsigned int index ) const
      {
         return at( index );
      }

      /**
       * Random member access operator.
       *
       * @param index the index of the object.
       *
       * @return reference the the object.
       */
      reference operator[](unsigned int index)
      {
         return at( index );
      }

      /**
       * Print all information about the vector.
       *
       */
      void Print() const
      {
         std::cout << "DataVector, capacity: " << m_capacity
                   << ", size: " << m_size << std::endl;
         for (size_type i = 0; i < m_capacity; ++i) {
            std::cout << m_first + i;
            if ( i < m_size ) {
               T* tmp = *( m_first + i );
               std::cout << " " << tmp;
               // if ( tmp ) {
               //    Reflex::Type type = Reflex::Type::ByTypeInfo( typeid( *tmp ) );
               //    std::cout << " " << type.Name( Reflex::SCOPED );
               // }
            }
            std::cout << std::endl;
         }
      }

      /**
       * Operator ==
       *
       * @param other the other DataVector
       *
       * @return is equal
       */
      bool operator==( const DataVector& other ) const
      {
         if (this == &other) return true;
         if (m_size == other.m_size && m_capacity == other.m_capacity) {
            for (size_type i = 0; i < m_size; ++i) {
               if (*(m_first + i) != *(other.m_first + i)) return false;
            }
         }
         return true;
      }

      /**
       * Operator !=
       *
       * @param other the other DataVector
       *
       * @return is not equal
       */
      bool operator!=( const DataVector& other ) const
      {
         if (this == &other) return false;
         return *this == other ? false : true;
      }

      virtual void Add( DataObject* data )
      {
         T* tmp  = dynamic_cast< T* >( data );
         if ( data && !tmp ) {
            return;
         } else {
            push_back( tmp );
         }
      }

      virtual const_reference At( unsigned int index ) const
      {
         return at( index );
      }

      virtual reference At( unsigned int index )
      {
         return at( index );
      }

      virtual void Release( DataObject* data )
      {
         if ( !data )  return;

         T* dataObject = dynamic_cast< T* >( data );
         typename addressMap_t::const_iterator it = m_addresses.find( dataObject );
         if ( it == m_addresses.end() ) return;

         data->SetContainer( 0 );
         size_type capacity = 0;
         size_type nCopyBottom = 0;
         size_type nCopyTop = 0;
         size_type size = m_size - 1;

         typename DataVector< T >::iterator left( m_first + it->second );
         typename DataVector< T >::iterator right = left + 1;

         nCopyBottom = distance( begin(), left );
         nCopyTop = distance( right, end() );
         if (size < 6) {
            capacity = 10;
         } else if (size < 2 * m_capacity / 3) {
            capacity = 2 * m_capacity / 3;
            if (capacity < 10) capacity = 10;
         } else {
            capacity = m_capacity;
         }

         pointer first = Allocator::allocate( capacity );
         size_type i;
         try {
            // Copy Bottom half
            for (i = 0; i < nCopyBottom; ++i) {
               Allocator::construct( first + i, *( m_first + i ) );
            }
            // Copy the top half
            for (i = 0; i < nCopyTop; ++i) {
               Allocator::construct( first + nCopyBottom + i, *( &*right + i ) );
            }
         }
         catch(...) {
            for(size_type j = 0; j < i; ++j) {
               Allocator::destroy( first + j );
            }
            Allocator::deallocate( first, capacity );
            throw;
         }
         // delete old stuff.
         m_addresses.erase( dataObject );

         for(size_type i = 0; i < m_size; ++i) {
            Allocator::destroy( m_first + i );
         }
         Allocator::deallocate(m_first, m_capacity);

         m_first = first;
         m_capacity = capacity;
         m_size = size;
         m_last = m_first + m_size;
      }
   };

}
#endif // DATAVECTOR_H
