/*
 *   File:      mytools.h
 *   Purpose:   My general tools and routines
 *   Author:    Jiri Kvita 
 *   Created:   15 Oct 2003
 *   Modified:  21 Feb 2004 - MyProfile
                29 Jun 2004 - FloatToChar and related conversions
                22 Jul 2004 - THisto changed to list of poiters
		27 Jul 2004 - new GetMyProfile, GetStrips
 */

#ifndef MYTOOLS
#define MYTOOLS

#include <TROOT.h>
#include <iostream.h>
#include <fstream.h> 
#include <math.h>
#include <list.h>
#include <TH1F.h>
#include <TH2F.h>
#include <TChain.h>
#include <TProfile.h>
#include <string>
#include <sstream>

#define abs(x) ( (x)>0 ? (x) : -(x) )

// ________________________________________________________________________
// Constants for histograms:)

const int maxBins=500;
const Float_t Pi=3.1415926536;
const int max_snprintf_char_size=500;
const int maxstrips=500;

const Float_t error=-9999;
const Float_t Zero=1.e-5;
const Int_t MinimumEntries=6;

//_________________ Courtesy of Robert Zitoun _____________________________   
class TEventTag
{
public:
    TEventTag(int krun, int kevt)
    {
        _run= krun;
        _evt= kevt;
    }
    int _run;
    int _evt;
};

// ________________________________________________________________________
class TCandList
{ 
 public:
  TCandList();
  bool IsAlreadyInList(int run, int evt);
 private:
  list<TEventTag> _CandList;
};
// ________________________________________________________________________
int ReadChain(char* inputfile,TChain* MyChain);
//{---------------------------------------------------------------------}
float Myatan(float x, float y);
//  ________________________________________________________________________
int min(int a, int b);
// ________________________________________________________________________
char* Uppercase(const char* tmp);
// ________________________________________________________________________
int Uppercase(char* tmp);
// ________________________________________________________________________
int IntToChar(char* str, int a);
// ________________________________________________________________________
int FloatToChar(char* str, float f);
// ________________________________________________________________________
bool CharToInt(int* ind, char* s);
// ________________________________________________________________________
void starline();
// ________________________________________________________________________
void dashline();
// ________________________________________________________________________
void fstarline(ofstream* fout);
// ________________________________________________________________________
void fdashline(ofstream* fout);
// ________________________________________________________________________
int GetStrips(TH2F* h2, TH1F** Strip, char* basename, char* datatype, int colour, int rebin);
// ________________________________________________________________________
float GetMedian(TH1F* histo);
// ________________________________________________________________________
float GetMedianError(TH1F* histo);
// ________________________________________________________________________
float GetSigma(TH1F* histo);
// ________________________________________________________________________
float GetMeanError(TH1F* histo);
// ________________________________________________________________________
bool AllEntriesInOneBin(TH1F* histo);
// ________________________________________________________________________
bool FitGauss(TH1F* histo, int rebin, float width, float* mean, float* sigma, float* error, const char* fitoption);
// ________________________________________________________________________
TH1F* GetMyProfile(TH2F* histo, char* suffix);
// ________________________________________________________________________
TH1F* GetMyProfile(TH2F* histo, char* suffix, const char* option);
// ________________________________________________________________________
TH1F* GetGaussProfile(TH2F* histo, char* suffix, int bins, int *rebin, float width);
// ________________________________________________________________________
TH1F* GetGaussProfile(TH2F* histo, char* suffix, int rebin, float width);
// ________________________________________________________________________
int RecomputeErrors(TH1F** h1, TH2F*h2);
// ________________________________________________________________________
float sqr(float x);
// ________________________________________________________________________
float GetdR(float eta1, float phi1, float eta2, float phi2);


// ________________________________________________________________________
// my modified histogram classes

// ________________________________________________________________________
class myTH1F  : public TH1F
{
 public:
  Bool_t IsEqualBinned();
};
// ________________________________________________________________________
class myTH2F  : public TH2F
{
 public:
  Bool_t IsEqualBinned();
};
// ________________________________________________________________________
class myTProfile  : public TProfile
{
 public:
  Bool_t IsEqualBinned();
};

// ________________________________________________________________________
// my class to handle histogramms TH1F and TH2F:)
// allows one to write all histograms using  Int_t THIsto::WriteAll();
// routines for filling the histogram according to its name and more:)


class THisto

{
 public:

  // constructor and destructor:)
  THisto();
  THisto(const char* name);
  THisto(const char* name, const char* option);
  ~THisto();
  
  // general methods

  char* GetName();
  Int_t DeleteAll();
  Int_t WriteAll();
  Int_t WriteHisto(char* name);
  bool Exist(char * name);

 // methods for TH1F histograms  
  Int_t AddH1(char* name,char* title, Int_t binsx, Float_t minx, Float_t maxx);
  Int_t AddH1(char* name,char* title, Int_t binsx, Float_t minx, Float_t maxx, char* option);
  Int_t AddH1(char* name,char* title, Int_t binsx, Float_t *xbins);
  Int_t AddH1(char* name,char* title, Int_t binsx, Float_t *xbins, char* option);

  Int_t NullH1(char* name);
  Int_t DeleteH1(char* name);
  TH1F* GetH1(char* name);
  Int_t FillH1(char* name, Float_t valuex);
  Int_t FillH1(char* name, Float_t valuex, Float_t weighty);
  TH1F* GetFirstH1();
  TH1F* GetLastH1();
  
  // methods for TH2F histograms      
  Int_t AddH2(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Int_t binsy, Float_t miny, Float_t maxy);
  Int_t AddH2(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Int_t binsy, Float_t miny, Float_t maxy, char* option);
  Int_t AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Float_t miny, Float_t maxy);
  Int_t AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Float_t miny, Float_t maxy, char* option);
  Int_t AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Double_t *ybins);
  Int_t AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Double_t *ybins, char* option);

  Int_t NullH2(char* name);
  Int_t DeleteH2(char* name);
  TH2F* GetH2(char* name);
  Int_t FillH2(char* name, Float_t valuex, Float_t valuey);
  Int_t FillH2(char* name, Float_t valuex, Float_t valuey, Float_t weightz);
  TH2F* GetFirstH2();
  TH2F* GetLastH2();

 // methods for TProfiles      
  Int_t AddProfile(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Float_t miny, Float_t maxy);
  Int_t AddProfile(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Float_t miny, Float_t maxy, char* option);
  Int_t AddProfile(char* name, char* title, Int_t binsx, Double_t *xbins, Float_t miny, Float_t maxy);
  Int_t AddProfile(char* name, char* title, Int_t binsx, Double_t *xbins, Float_t miny, Float_t maxy, char* option);
  Int_t DeleteProfile(char* name);
  TProfile* GetProfile(char* name);
  Int_t FillProfile(char* name, Float_t valuex, Float_t valuey);
  Int_t FillProfile(char* name, Float_t valuex, Float_t valuey, Float_t weightz);
  TProfile* GetFirstProfile();
  TProfile* GetLastProfile();

  int Sumw2(char* name);

 private:
  list<TH1F*> _H1;
  list<TH2F*> _H2;
  list<TProfile*> _Profile;
  bool _verbose;
  char* _name;
};


// ________________________________________________________________________

#endif
