/*
 *   File:      mytools.cpp
 *   Purpose:   My general tools and routines
 *   Author:    Jiri Kvita 
 *   Created:   20 Oct 2003
 *   Modified:  $Id: mytools.cpp,v 1.9 2004/10/07 22:25:52 kvita Exp $
 */

#include <TROOT.h>
#include <list.h>
#include "mytools.h"
#include <TF1.h>
#include <TArrayD.h>
#include <iostream.h>
#include <fstream.h> 
#include <sstream>
#include <string>

using namespace std;

// ________________________________________________________________________

int ReadChain(char* inputfile,TChain* MyChain)
{
  list<string> files;  
  ifstream f;
  f.open(inputfile);
  if (f.is_open()) {
    string s;
    dashline();
    // inserting all files into the chain
    while (f>>s) if ((! f.fail()) && (s.size())) {
      const char* name = (s.c_str());
      MyChain->AddFile(name,-1);
      //if nentries <= 0, the file is connected and the tree header read in memory to get the number of entries.
      cout << "File " << name << " added to the chain.\n";
      files.push_front(s);
    }  
    f.close();
    return 1;
  } else {
    cout << "Check file list name, " << inputfile << " couldn't be found!\n"; 
    return 0; 
  }
}
// ________________________________________________________________________

int min(int a, int b)
{
  if (a<b) return a; else return b;
}
// ________________________________________________________________________

float Myatan(float x, float y)
//function returning angle in range (0,2*Pi)
{
  float phi=atan(y/x);
  if (x>0) {if (y<0) return 2*Pi+phi;/*phi is negative here! */ }
  else if (y<0) return Pi+phi;
  else return Pi+phi; /* phi negative here as well!*/
  return phi;
}
// ________________________________________________________________________

char* Uppercase(const char* tmp)
{
  int len=strlen(tmp);
  char* newtmp;
  newtmp = new char[len+1];
  strcpy(newtmp, tmp);
  for(int i=0; i<len; ++i) newtmp[i]=toupper(tmp[i]);
  return newtmp;
}
// ________________________________________________________________________

int Uppercase(char* tmp)
{
  for(int i=0; tmp[i]; ++i) tmp[i]=toupper(tmp[i]);
  return strlen(tmp);
}
// ________________________________________________________________________

bool CharToInt(int* ind, char* s)
{ 
  int help=0;
  for(int i=0; i<strlen(s); i++)
    {
      // cast to ints and subtract 48 to account for ascii values of digits
      help=help*10;
      if (((int)s[i] - 48<0)||((int)s[i] - 48>9)) return false;
      help=help+(int)s[i] - 48;
    }
  *ind=help;
  return true;
}
// ________________________________________________________________________

int IntToChar(char* str, int a)
{
  //  ostringstream* sin;
  //sin = new ostringstream(); 
  //(*sin) << a;
  //string ss="damned";//sin->str();
  //  delete sin;
  //return ((char*) ss.c_str());

  if (!str) str = new char[max_snprintf_char_size];
  strcpy(str,""); 
  snprintf(str,max_snprintf_char_size,"%i",a); 
  return 1;
}
// ________________________________________________________________________

int FloatToChar(char* str, float f)
{
  //ostringstream in1; 
  // in1  << f;
  //string ss;//=in1.str();
  //return (char*) ss.c_str()

  if (!str) str = new char[max_snprintf_char_size];
  strcpy(str,""); 
  snprintf(str,max_snprintf_char_size,"%f",f); 
  return 1;
}
// ________________________________________________________________________

void starline()
{
 cout << "******************************************************************************\n";
}
// ________________________________________________________________________

void dashline()
{
 cout << "+----------------------------------------------------------------------------+\n";
}
// ________________________________________________________________________

void fstarline(ofstream* fout)
{
 (*fout) << "******************************************************************************\n";
}
// ________________________________________________________________________

void fdashline(ofstream* fout)
{
  (*fout) << "+----------------------------------------------------------------------------+\n";
}
// ________________________________________________________________________

bool TCandList::IsAlreadyInList(int run, int evt)
{
  // Check if the event is in the list; else insert
    TEventTag *NewTag = new TEventTag(run, evt);
    list<TEventTag>::iterator iter = _CandList.begin();
    while (iter != _CandList.end())
    {
      if (iter->_evt==evt && iter->_run==run) return true;
      ++iter;
    }
    _CandList.push_back(*NewTag);
    delete NewTag;
    return false;
}
// ________________________________________________________________________

TCandList::TCandList()
{
}

// ________________________________________________________________________

int GetStrips(TH2F* h2, TH1F** Strip, char* basename, char* datatype, int colour, int rebin)
{

  //works only for equaly spaced bins since I need to rebin time to time:)
  if (h2) {
    float miny=h2->GetYaxis()->GetXmin();
    float maxy=h2->GetYaxis()->GetXmax();
    int  binsx=h2->GetXaxis()->GetNbins();
    int  binsy=h2->GetYaxis()->GetNbins(); 
 
    /*
      const TArrayD *Aybins =  h2->GetYaxis()->GetXbins();
    */
   
    //BEWARE! strips are filled starting at index 1!
    for(int kk=1;kk<=binsx;++kk) {
      char name[300],title[300];
      sprintf((char*)name,"%s_%4.2f,%4.2f_%s",basename,h2->GetBinLowEdge(kk),
	      h2->GetBinLowEdge(kk)+h2->GetBinWidth(kk),datatype); 
      sprintf((char*)title,"%4.2f < %s < %4.2f, %s",h2->GetBinLowEdge(kk),basename,
	      h2->GetBinLowEdge(kk)+h2->GetBinWidth(kk),datatype);
      Strip[kk] = new TH1F(name,title,binsy,miny,maxy);
      //      Strip[kk] = new TH1F(name,title,binsy,Aybins->fArray);
      int entr=0;
      //histogram filled within the range of y axis, so no over/under flows...
      //for(int ll=h2->GetYaxis()->GetFirst();ll<=h2->GetYaxis()->GetLast();++ll) {
      //including over and under flows:
      for(int ll=0;ll<=1+h2->GetNbinsY();++ll) {
	Strip[kk]->SetLineColor(colour);
	int llbin=h2->GetBinContent(kk,ll);
	Strip[kk]->SetBinContent(ll,llbin);
	entr+=llbin;
      }
      Strip[kk]->SetEntries(1.*entr);
      if (rebin>1) Strip[kk]->Rebin(rebin);
    }
  }//if (h2)

}
// ________________________________________________________________________

float GetMedian(TH1F* histo)
// I'm, using low bin edge instead of bin centre for shifting, 
// and finally I put median into the centre of the bin where number of cumulating
// events exceeds 1/2 of the integral within the zoomed axis range

//my median, works for weighted histogram and also for nonequaly spaced binning
//median is computed only from the range given by the X axis!

{
  if (histo) 
    {
      if (!(histo->GetEntries()>Zero)) return 0.;
      float median=1.*((histo->GetXaxis()->GetXmin())), sum=0;
      //if option "width" is specified to TH1F::Integral(), the integral is the sum of
      // the bin contents multiplied by the bin width in x, but that's what I don't want here
      float evts_integral=histo->Integral(histo->GetXaxis()->GetFirst(), histo->GetXaxis()->GetLast());
      if (!(evts_integral > Zero)) return 0.;
      //      cout << "--------- Computing median for "<<histo->GetName()<<" ---------\n";
      //cout <<"User range integral: "<<evts_integral<<"  Starting value for median: "<< median<<"  Full range integral: "<<histo->Integral()<<" Nbins=" <<histo->GetNbinsX() << endl;
      for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i)
	{
	  sum+=histo->GetBinContent(i);
	  // cout << i << ". bin content: " << histo->GetBinContent(i) << " actual median value: " << median << endl;
	  //
	  if (sum>=(evts_integral/2.)) {
	    median+=(histo->GetBinWidth(i))/2.;
	    //    cout << "half sum: " << sum-histo->GetBinContent(i) << " Mean: " << histo->GetMean() << "   Median: " << median<< endl;
	    return median;
	  }
	  median+=histo->GetBinWidth(i);
	}
    }
  return 0.;
}
// ________________________________________________________________________

float GetMedianError(TH1F* histo)
// L1 norm computing of the median error divided by sqrt(N) to match mean error definition
// what about single occupied bin?
{
  if (histo) 
    {

      if (!(histo->GetEntries()>Zero)) return 0.;
      float median=GetMedian(histo);
      float evts_integral=histo->Integral(histo->GetXaxis()->GetFirst(), histo->GetXaxis()->GetLast());
      if (!(evts_integral > Zero)) return 0.;

      //check wheter all entries are not one bin:
      for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i)
	if (evts_integral == histo->GetBinContent(i))  return (histo->GetBinWidth(i))/sqrt(histo->GetBinContent(i));
      //	  cout <<"====> In GetMedianError: all entries found in one bin; entries=" << histo->GetEntries() << endl;

      float sum=0;
      //computed without over and underflows
      for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i)
	sum+=(histo->GetBinContent(i))*fabs(histo->GetBinCenter(i)-median)/evts_integral;
      return sum/sqrt(evts_integral);
    }
  return 0.;
}
// ________________________________________________________________________

float GetSigma(TH1F* histo)
{
  //this returs the sigma of the distribution, not the error of mean!
  //returns the same number as TH1::GetRMS(), which is a historical name, routine gives sigma
  if (histo) 
    {
      float mean=histo->GetMean(), sum=0;
      float evts_integral=histo->Integral(histo->GetXaxis()->GetFirst(), histo->GetXaxis()->GetLast());
      if (!(evts_integral > 0)) return 0.;
      //computed without over and underflows
      for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i) {
	sum+=(histo->GetBinContent(i))*(histo->GetBinCenter(i)-mean)*(histo->GetBinCenter(i)-mean);
	//if all events in one bin, assume poisson error:
	if (histo->GetBinContent(i) == evts_integral) return sqrt(evts_integral);
      }
      return sqrt(sum/(evts_integral));
    }
  return 0.;
}
// ________________________________________________________________________

float GetMeanError(TH1F* histo)
{
  //this returs the error of mean, i.e. sigma/sqrt(N)!
  if (histo) 
    {
      if (!(histo->GetEntries()>Zero)) return 0.;

      float sigma=GetSigma(histo);
      float evts_integral=histo->Integral(histo->GetXaxis()->GetFirst(), histo->GetXaxis()->GetLast());

      //check whether nentries!=0
      if (!(evts_integral > 0)) {
	return 0.;
      }
      
      //check wheter all entries are not one bin:
      for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i)
	if (evts_integral == histo->GetBinContent(i)) 
	  return (histo->GetBinWidth(i))/sqrt(histo->GetBinContent(i));
      
      if (evts_integral>Zero) return sigma/sqrt(evts_integral);
    }
  return 0.;
}


// ________________________________________________________________________
bool AllEntriesInOneBin(TH1F* histo)
{
  float evts_integral=histo->Integral(histo->GetXaxis()->GetFirst(), histo->GetXaxis()->GetLast());
  
  //check whether nentries!=0
  if (!(evts_integral > 0)) {
    return false;
  }
  //check wheter all entries are not one bin:
  for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i)
    if (evts_integral == histo->GetBinContent(i)) 
      return true;
  return false;
}

// ________________________________________________________________________

bool FitGauss(TH1F* histo, int rebin, float width, float* mean, float* sigma, float* error, const char* fitoption)
//performs a gauss fit on the interval (mean-width*sigma,mean+width*sigma)
{
  if (histo)
    {
      if (rebin>0) histo->Rebin(rebin);
      float hmean = histo->GetMean();
      float hsigma = GetSigma(histo);
      
      //check whether nentries!=0
      float evts_integral=histo->Integral(histo->GetXaxis()->GetFirst(), histo->GetXaxis()->GetLast());
      if  (!(evts_integral > Zero)) {
	*error=0; *sigma=0; *mean=0;
	return 1;
      }
      
      if  (evts_integral < MinimumEntries) {
	*sigma=hsigma; 
	*mean=hmean;
	*error=(*sigma)/sqrt(evts_integral);
	cout << "Too few entries to perform Gauss fit! Nevts=" << evts_integral << "!\n";
	return 1;
      }
      
      //check wheter all entries are not one bin:
      for(int i=histo->GetXaxis()->GetFirst();i<=(histo->GetXaxis()->GetLast());++i)
	if (evts_integral == histo->GetBinContent(i)) {
	  *sigma=(histo->GetBinWidth(i)); 
	  *mean=hmean;
	  *error=(*sigma)/sqrt(histo->GetBinContent(i));
	  cout << "All entries in one bin i=" << i << "!\n";
	  return 0;
	}
      
      if (width<0.) {
	cout << "Requested interval size in units of sigma smaller than zero, I'm multiplying it by -1.\n";
	width*=(-1);
      }
      TF1 g("xmygauss","[0]*exp(-(x-[1])^2/(2*[2]^2))",hmean-width*hsigma,hmean+width*hsigma);
      g.SetParameters(histo->GetMaximum(),hmean,hsigma);
      g.SetParNames("A","#mu","#sigma");
      g.SetParLimits(2,0.,100.*(histo->GetXaxis()->GetXmax() - histo->GetXaxis()->GetXmin()));
      g.SetLineWidth(1);
      histo->Fit(&g,fitoption);
      *sigma = g.GetParameter(2);
      *mean = g.GetParameter(1);
      //number of entries within fit range:
      float neff = histo->Integral(histo->FindBin(hmean-width*hsigma),histo->FindBin(hmean+width*hsigma));
      if (neff > 0) *error = (*sigma)/(sqrt(neff));
      else *error = 0.;
      char out[1000];
      sprintf(out,"A=%6.3f, mean=%6.3f, sigma=%6.3f, Neff=%6.3f, error=%8.5f",g.GetParameter(0),*mean,*sigma,neff,*error);
      cout << out << endl;
      //"A="<<g.GetParameter(0)<<", mean="<<(*mean)<<", sigma="<<(*sigma)<<", Neff="<<neff<<", error="<<(*error)<<endl;
      return 1;
    }
  cout << "Can't fit requested histogram, got a null pointer!\n";
  return 0;
}
// ________________________________________________________________________

TH1F* GetMyProfile(TH2F* histo, char* suffix, const char* option)
  //my own profile from TH2F; GAUSS, MEAN or MEDIAN options can be specified:)
{

  if (!strcmp(Uppercase(option),"GAUSS"))  return GetGaussProfile(histo, suffix, 1, 2.);
  if (histo) {
    int Xbins = histo->GetXaxis()->GetNbins();
    //int Ybins = histo->GetYaxis()->GetNbins();

    const TArrayD *Axbins =  histo->GetXaxis()->GetXbins();
    //    Double_t xbins[maxBins];
    //if (!(Axbins->fN == 0)) for(int i=0;i<Xbins;++i) xbins[i] = (Axbins->At(i));

    float maxX = histo->GetXaxis()->GetXmax();
    float minX = histo->GetXaxis()->GetXmin();
    float maxY = histo->GetYaxis()->GetXmax();
    float minY = histo->GetYaxis()->GetXmin();

    char newname[500]; strcpy(newname,"");strcpy(newname,histo->GetName());
    strcat(newname,"_");strcat(newname,option);strcat(newname,"_prof");strcat(newname,suffix);
    char newtitle[500];strcpy(newtitle,"");strcpy(newtitle,histo->GetTitle());
    strcat(newtitle,"_");strcat(newtitle,option);strcat(newtitle,"_prof");
    TH1F *h1;
    if (!(Axbins->fN == 0)) 
         h1 = new TH1F((char*)newname,(char*)newtitle,Xbins,Axbins->fArray);
    else h1 = new TH1F((char*)newname,(char*)newtitle,Xbins,minX,maxX);

    h1->Sumw2();
    h1->SetMaximum(maxY);
    h1->SetMinimum(minY);

    TH1F* strip[maxstrips];
    char* basename="Strip";
    char* tag="tmp";
    GetStrips(histo,(TH1F**)strip,(char*)basename,(char*)tag,1,1);
    if (!strcmp(Uppercase(option),"MEDIAN"))
      {
	//median based profile
	for(int i=1; i<=Xbins; ++i) 
	  if (strip[i]->GetEntries()>Zero) {
	    //	    cout << "Strip entries="<<strip[i]->GetEntries()<<endl;
	    h1->SetBinContent(i,1.*GetMedian(strip[i]));
	    h1->SetBinError(i,1.*GetMedianError(strip[i]));
	  }
      }
	else
	  {
	    //mean based profile, default
	    for(int i=1; i<=Xbins; ++i) 
	      if (strip[i]->GetEntries()>Zero) {
		h1->SetBinContent(i,1.*(strip[i]->GetMean()));
		h1->SetBinError(i,1.*GetMeanError(strip[i]));
	      }
	  }


    bool OneBin[maxstrips];
    for(int r = 0; r<maxstrips; ++r) OneBin[r]=false;
    int nOneBin=0;
    int nOK=0;
    float errsum=0.;
    for(int i=1; i<=Xbins; ++i) {
      if ( (AllEntriesInOneBin(strip[i])) || 
	   ( (strip[i]->GetEntries()<MinimumEntries) && (strip[i]->GetEntries()>Zero)) ) {
	OneBin[i]=true;
	nOneBin++;
      } else 
	if (h1->GetBinError(i)>Zero) {
	  errsum+= h1->GetBinError(i);
	  nOK++;
	}
    }
  

    cout << "Found " << nOK <<" OK bins, " << nOneBin << " bins with all events in one bin.\n";
    if (nOK>Zero) {
      float err=2.*errsum/(1.*nOK);
      cout << "Average error over OK bins: " << errsum/(1.*nOK) << endl;
      for(int i=1; i<=Xbins; ++i) {
	if (OneBin[i]) {
	  h1->SetBinError(i,err);
	} 
      }
    }

    h1->SetEntries(histo->GetEntries());




    for(int i=1; i<=Xbins; ++i) delete (strip[i]);
    return h1;
  }
  else return 0;
}
// ________________________________________________________________________

TH1F* GetMyProfile(TH2F* histo, char* suffix)
{
  //by default returns mean based profile
  return GetMyProfile(histo, suffix, "mean");
}
// ________________________________________________________________________

TH1F* GetGaussProfile(TH2F* histo, char* suffix, int rebin, float width)
{
  return GetGaussProfile(histo, suffix, 1, &rebin, width);
}
// ________________________________________________________________________

TH1F* GetGaussProfile(TH2F* histo, char* suffix, int bins, int *rebin, float width) 
//gauss profile with variable n-sigma interval for fitting!
{
 if (histo) {
    int Xbins=histo->GetXaxis()->GetNbins();
    //int Ybins=histo->GetYaxis()->GetNbins();

    const TArrayD *Axbins =  histo->GetXaxis()->GetXbins();
    //Double_t xbins[maxBins];
    //(!(Axbins->fN == 0)) for(int i=0;i<Xbins;++i) xbins[i] = (Axbins->At(i));

    float maxX=histo->GetXaxis()->GetXmax();
    float minX=histo->GetXaxis()->GetXmin();
    float maxY=histo->GetYaxis()->GetXmax();
    float minY=histo->GetYaxis()->GetXmin();

    char newname[500]; strcpy(newname,"");strcpy(newname,histo->GetName());
    strcat(newname,"_");strcat(newname,"gauss");strcat(newname,"_prof");strcat(newname,suffix);
    char newtitle[500];strcpy(newtitle,"");strcpy(newtitle,histo->GetTitle());
    strcat(newtitle,"_");strcat(newtitle,"gauss");strcat(newtitle,"_prof");
    TH1F* h1; 
    if (!(Axbins->fN == 0)) h1 = new TH1F((char*)newname,(char*)newtitle,Xbins,Axbins->fArray);
    else  h1 = new TH1F((char*)newname,(char*)newtitle,Xbins,minX,maxX);
    cout << "Created a gauss profile " << h1->GetName() << ", will fit now..." << endl;
    h1->Sumw2();
    h1->SetMaximum(maxY);
    h1->SetMinimum(minY);

    TH1F* strip[maxstrips];
    char* basename="Strip";
    char* tag="tmp";
    GetStrips(histo,(TH1F**)strip,(char*)basename,(char*)tag,1,1);
    float gmean=0,gsigma=0,gerror=0;
    int _rebin=1;
    for(int i=1; i<=Xbins; ++i) {
      if (bins>0) {
	if (i <= bins) {
	  if (rebin[i-1] > 0)
	    _rebin = rebin[i-1]; 
	}
	else {
	  if (rebin[bins-1] > 0)
	    _rebin = rebin[bins-1];
	}
      }
      cout<< "Fitting bin #" << i << ", rebinning by factor "<<_rebin<<"\n";
      if (FitGauss(strip[i], _rebin, width, &gmean, &gsigma, &gerror,"Q0"))
	if (strip[i]->GetEntries()>Zero) 
	  {
	    h1->SetBinContent(i,gmean);
	    h1->SetBinError(i,gerror);
	  }
    }//for
    cout << endl;


    bool OneBin[maxstrips];
    for(int r = 0; r<maxstrips; ++r) OneBin[r]=false;
    int nOneBin=0;
    int nOK=0;
    float errsum=0.;
    for(int i=1; i<=Xbins; ++i) {
      if ( (AllEntriesInOneBin(strip[i])) ||  
	   ( (strip[i]->GetEntries()<MinimumEntries) && (strip[i]->GetEntries()>Zero)) ) {
	OneBin[i]=true;
	nOneBin++;
      } else 
	if (h1->GetBinError(i)>Zero) {
	  errsum+= h1->GetBinError(i);
	  nOK++;
	}
    }
    
    cout << "Found " << nOK <<" OK bins, " << nOneBin << " bins with all events in one bin.\n";
    if (nOK>Zero) {
      float err=2.*errsum/(1.*nOK);
      cout << "Average error over OK bins: " << errsum/(1.*nOK) << endl;
      for(int i=1; i<=Xbins; ++i) {
	if (OneBin[i]) {
	  h1->SetBinError(i,err);
	} 
      }
    }

  h1->SetEntries(histo->GetEntries());
    for(int i=1; i<=Xbins; ++i) delete (strip[i]);
    return h1;
  }
  else return 0;
}

// ________________________________________________________________________

int RecomputeErrors(TH1F** h1, TH2F* h2)
{
  //shouldn't be used, needs revision!
  //recomputing errors of a (typically) profile created from a TH2F filled with weights
  //works only for equaly spaced bins!
 if ((*h1)&&(h2)) {
    int Xbins=h2->GetNbinsX();
    int Ybins=h2->GetNbinsY();
    float maxX=h2->GetXaxis()->GetXmax();
    float minX=h2->GetXaxis()->GetXmin();
    float maxY=h2->GetYaxis()->GetXmax();
    float minY=h2->GetYaxis()->GetXmin();
    cout << "Histo 2D name: "<< h2->GetName()<<endl;
    for(int i=1; i<=Xbins; ++i) {
      float aver=0,nomin=0,evts=0;
      float step=(maxY-minY)/(1.*Ybins);
      float y=minY+step/2.;
      for(int j=1; j<=Ybins; ++j) {
	aver+=(h2->GetBinContent(i,j))*y;
	evts+=(h2->GetBinContent(i,j));
	//cout << i<<" "<<j<<" ... " <<h2->GetBinContent(i,j)<<endl; 
	y+=step;
      }
      if (evts==0) {
	cout << "Setting errors of "<<(*h1)->GetName() << " to zero!\n";
	(*h1)->SetBinError(i,0);
	continue;
      }
      aver=aver/evts;
      float err=0;
      y=minY+step/2.;
      for(int j=1; j<=Ybins; ++j) {
	nomin+=(aver-y)*(aver-y)*(h2->GetBinContent(i,j))/evts;
	y+=step;
      }
      err=sqrt(nomin/evts);
      (*h1)->SetBinError(i,err);
    }
    return 1;
  }
  else return 0;
}
// ________________________________________________________________________

float sqr(float x)
{
  return x*x;
}
// ________________________________________________________________________

float GetdR(float eta1, float phi1, float eta2, float phi2)
//returns dR of two calorimeter objects
{
  return sqrt( sqr(eta1-eta2) + sqr(phi1-phi2) );
}
// ________________________________________________________________________
// my modified histogram classes

// ________________________________________________________________________

Bool_t myTH1F::IsEqualBinned()
{
  const TArrayD *bins = fXaxis.GetXbins();
  return (bins->fN == 0);
}
// ________________________________________________________________________

Bool_t myTH2F::IsEqualBinned()
{
 const TArrayD *bins = fXaxis.GetXbins();
 return (bins->fN == 0);
}
// ________________________________________________________________________

Bool_t myTProfile::IsEqualBinned()
{
 const TArrayD *bins = fXaxis.GetXbins();
 return (bins->fN == 0);
}

// ________________________________________________________________________
// ________________________________________________________________________
// ________________________________________________________________________

THisto::THisto()
{
  char* name="Histograms";
  int len=strlen(name);
  len=strlen(name);
  _name = new char[len+1];
  strcpy(_name,name);
  _verbose=false;
   if (_verbose) cout << "THisto "<< _name <<" inicialized with no option." << endl;
}    
// ________________________________________________________________________

THisto::THisto(const char* name, const char* option)
{

  int len=strlen(option);
  char* tmp = new char[len+1];
  strcpy(tmp,option);

  Uppercase(tmp);
  if (!strcmp(tmp,"V")) _verbose=true;
  if (!strcmp(tmp,"Q")) _verbose=false;
  
  len=strlen(name);
  _name = new char[len+1];
  strcpy(_name,name);

  if (_verbose) cout << "THisto "<< _name <<" inicialized with option "<<tmp << endl;

}    
// ________________________________________________________________________
THisto::THisto(const char* name)
{
  //minimum of output when creating histograms
  _verbose=false;
  int len=strlen(name);
  _name = new char[len+1];
  strcpy(_name,name);

}
// ________________________________________________________________________

THisto::~THisto()
{
  DeleteAll();
}
// ________________________________________________________________________

char* THisto::GetName()
{
  return _name;
}

// ________________________________________________________________________

Int_t THisto::DeleteAll()
{
  for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (*_h1) delete (*_h1);
    }

  for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (*_h2) delete (*_h2);
    }
  
  for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if (*_profile) delete (*_profile);
    }
  _H1.clear();
  _H2.clear();
  _Profile.clear();
  return 1;
}
// ________________________________________________________________________

Int_t THisto::WriteAll()
{
  for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      (*_h1)->Write();
    }

  for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      (*_h2)->Write();
    }
  
  for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      (*_profile)->Write();
    }

  return 1;
}
// ________________________________________________________________________

Int_t THisto::WriteHisto(char* name)
{

 for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) {(*_h1)->Write(); return 1; }
    }

  for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if ((*_h2)->GetName(),name) {(*_h2)->Write(); return 1;}
    }

  for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if ((*_profile)->GetName(),name) {(*_profile)->Write(); return 1;}
    }

  return 0;
}
// ________________________________________________________________________

bool THisto::Exist(char* name)
{
 bool _h1exists=0,_h2exists=0, _profileexists=0;

  for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) {_h1exists=1;}
    }

  for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (!strcmp((*_h2)->GetName(),name)) {_h2exists=1; }
    }

 for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
   {
     if (!strcmp((*_profile)->GetName(),name)) {_profileexists=1; }
    }

  return (_h1exists || _h2exists || _profileexists);
}
// ________________________________________________________________________
//
// methods for TH1F histograms      
// ________________________________________________________________________

Int_t THisto::AddH1(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx)
{
  return AddH1(name,title,binsx, minx, maxx,"");
}
// ________________________________________________________________________

Int_t THisto::AddH1(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, char* option)
{
  if (!Exist(name)) 
    {
      //      cout << "Creating a new histogram..";
      TH1F *h1 = new TH1F(name,title,binsx, minx, maxx);
      _H1.push_back(h1);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	h1->Sumw2();
	if (_verbose) cout << "Sumw2() method called for histogram named " << name << endl;
      }
      //cout << ".created!\n";
      return 1;
    }
  else 
    {
      cout << "TH1F histogram " << name << " already exists, won't be replaced!\n";
      return 0;
    }
}

// ________________________________________________________________________

Int_t THisto::AddH1(char* name,char* title, Int_t binsx, Float_t *xbins, char* option)
// xbins  : array of low-edges for each bin
//          This is an array of size nbins+1
{
 if (!Exist(name)) 
    {
      //      cout << "Creating a new histogram..";
      TH1F *h1 = new TH1F(name,title,binsx, xbins);
      _H1.push_back(h1);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	h1->Sumw2();
	if (_verbose) cout << "Sumw2() method called for histogram named " << name << endl;
      }
      //cout << ".created!\n";
      return 1;
    }
  else 
    {
      cout << "TH1F histogram " << name << " already exists, won't be replaced!\n";
      return 0;
    }
}
// ________________________________________________________________________

Int_t THisto::AddH1(char* name,char* title, Int_t binsx, Float_t *xbins)
{
  return AddH1(name,title,binsx,xbins,"");
}
// ________________________________________________________________________

Int_t THisto::NullH1(char* name)
{
  TH1F *histo=GetH1(name);
  if (histo)
    {
      for(int i=0;i<histo->GetNbinsX();++i) {
	histo->SetBinContent(i,0);
	histo->SetBinError(i,0);
      }
      histo->SetEntries(0.);
      return 1;
    }
  else return 0;
}

// ________________________________________________________________________

Int_t THisto::DeleteH1(char* name)
{
  for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) 
	{ 
	  if ((*_h1)) delete (*_h1);
	  _H1.erase(_h1); return 1; 
	}
    }
  return 0;
}
// ________________________________________________________________________

TH1F* THisto::GetH1(char* name)
{
  // To be checked!!! 
 for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) {return (*_h1);}
    }
  return 0;
}
// ________________________________________________________________________

Int_t THisto::FillH1(char* name, Float_t valuex)
{
 for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) 
	{ (*_h1)->Fill(valuex); return 1; }
    }
 cout << "Tried to fill nonexisting TH1F " << name << "!\n";
  return 0;
}
// ________________________________________________________________________

Int_t THisto::FillH1(char* name, Float_t valuex, Float_t weighty)
{
 for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) 
	{ (*_h1)->Fill(valuex, weighty); return 1; }
    }
 cout << "Tried to fill nonexisting TH1F " << name << "!\n";
  return 0;
}
// ________________________________________________________________________

TH1F* THisto::GetFirstH1()
{
  return (_H1.front());
}
// ________________________________________________________________________

TH1F* THisto::GetLastH1()
{
  return (_H1.back());
}
// ________________________________________________________________________
//
// methods for TH2F histograms      
// ________________________________________________________________________

Int_t THisto::AddH2(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Int_t binsy, Float_t miny, Float_t maxy)
{
  return AddH2(name,title,binsx, minx, maxx, binsy, miny, maxy, "");
}
// ________________________________________________________________________

Int_t THisto::AddH2(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Int_t binsy, Float_t miny, Float_t maxy, char* option)
{
  if (!Exist(name)) 
    {
      
      TH2F *h2 = new TH2F(name,title,binsx, minx, maxx, binsy, miny, maxy);
      _H2.push_back(h2);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	h2->Sumw2();
	if (_verbose) cout << "Sumw2() method called for histogram named " << name << endl;
      }
      return 1;
    }
  else 
    {
      cout << "TH2F histogram " << name << " already exists, won't be replaced!\n";
      return 0;
    }
}

// ________________________________________________________________________

Int_t THisto::AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Float_t miny, Float_t maxy, char* option)
{
if (!Exist(name)) 
    {
      
      TH2F *h2 = new TH2F(name, title, binsx, xbins, binsy, miny, maxy);
      _H2.push_back(h2);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	h2->Sumw2();
	if (_verbose) cout << "Sumw2() method called for histogram named " << name << endl;
      }
      return 1;
    }
  else 
    {
      cout << "TH2F histogram " << name << " already exists, won't be replaced!\n";
      return 0;
    }
}
// ________________________________________________________________________

Int_t THisto::AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Float_t miny, Float_t maxy)
{
  return AddH2(name, title, binsx, xbins, binsy, miny, maxy,"");
}

// ________________________________________________________________________

Int_t THisto::AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Double_t *ybins, char* option)
{
  if (!Exist(name)) 
    {
      TH2F *h2 = new TH2F(name, title, binsx, xbins, binsy, ybins);
      _H2.push_back(h2);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	h2->Sumw2();
	if (_verbose) cout << "Sumw2() method called for histogram named " << name << endl;
      }
      return 1;
    }
  else 
    {
      cout << "TH2F histogram " << name << " already exists, won't be replaced!\n";
      return 0;
    }
}
// ________________________________________________________________________

Int_t THisto::AddH2(char* name, char* title, Int_t binsx, Double_t *xbins, Int_t binsy, Double_t *ybins)
{
  return AddH2(name, title, binsx, xbins, binsy, ybins, "");
}
// ________________________________________________________________________

Int_t THisto::NullH2(char* name)
{
  TH2F *histo=GetH2(name);
  if (histo)
    {
      for(int i=0;i<histo->GetNbinsX();++i) 
	for(int j=0;j<histo->GetNbinsY();++j) {
	  histo->SetBinContent(i,j,0);
	  histo->SetBinError(i,j,0);
	}
      return 1;
    }
  else return 0;
}
// ________________________________________________________________________

Int_t THisto::DeleteH2(char* name)
{
  for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (!strcmp((*_h2)->GetName(),name))
	{ 
	  if ((*_h2)) delete (*_h2);
	  _H2.erase(_h2); 
	  return 1;
	}
    }
  return 0;
}
// ________________________________________________________________________

TH2F* THisto::GetH2(char* name)
{
  // To be checked!!! 
  for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (!strcmp((*_h2)->GetName(),name)) {return (*_h2);}
    }
  return 0;  
}
// ________________________________________________________________________

Int_t THisto::FillH2(char* name, Float_t valuex, Float_t valuey)
{
 for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (!strcmp((*_h2)->GetName(),name)) 
	{ (*_h2)->Fill(valuex, valuey); return 1; }
    }
 cout << "Tried to fill nonexisting TH2F " << name << "!\n";
  return 0;
}
// ________________________________________________________________________

Int_t THisto::FillH2(char* name, Float_t valuex, Float_t valuey, Float_t weightz)
{
 for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (!strcmp((*_h2)->GetName(),name)) 
	{ (*_h2)->Fill(valuex, valuey, weightz); return 1; }
    }
 cout << "Tried to fill nonexisting TH2F " << name << "!\n";
 return 0;
}
// ________________________________________________________________________

TH2F* THisto::GetFirstH2()
{
  return (_H2.front());
}
// ________________________________________________________________________

TH2F* THisto::GetLastH2()
{
 return (_H2.back());
}
//methods for TProfiles:
// ________________________________________________________________________

Int_t THisto::AddProfile(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Float_t miny, Float_t maxy)
{
  return AddProfile(name,title,binsx,minx,maxx,miny,maxy,"");
}
// ________________________________________________________________________

Int_t THisto::AddProfile(char* name, char* title, Int_t binsx, Float_t minx, Float_t maxx, Float_t miny, Float_t maxy, char* option)
{
  if (!Exist(name)) 
    {
      TProfile *profile = new TProfile(name,title,binsx, minx, maxx, miny, maxy);
      _Profile.push_back(profile);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	profile->Sumw2();
	if (_verbose) cout << "Sumw2() method called for TProfile named " << name << endl;
      return 1;
      }
    }
      else 
	{
	  cout << "TProfile " << name << " already exists, won't be replaced!\n";
	  return 0;
	}
}

// ________________________________________________________________________

Int_t THisto::AddProfile(char* name, char* title, Int_t binsx, Double_t *xbins, Float_t miny, Float_t maxy, char* option)
{
 if (!Exist(name)) 
    {
      TProfile *profile = new TProfile(name, title, binsx, xbins, miny, maxy,option);
      _Profile.push_back(profile);
      char tmp[80];
      strcpy(tmp,"");
      strcpy(tmp,option);
      Uppercase(tmp);
      if (!strcmp(tmp,"SUMW2")) {
	profile->Sumw2();
	if (_verbose) cout << "Sumw2() method called for TProfile named " << name << endl;
      return 1;
      }
    }
      else 
	{
	  cout << "TProfile " << name << " already exists, won't be replaced!\n";
	  return 0;
	}
}
// ________________________________________________________________________

Int_t THisto::AddProfile(char* name, char* title, Int_t binsx, Double_t *xbins, Float_t miny, Float_t maxy)
{
  return AddProfile(name,title,binsx,xbins,miny,maxy,"");
}
// ________________________________________________________________________

Int_t THisto::DeleteProfile(char* name)
{
  for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if (!strcmp((*_profile)->GetName(),name))
	{ 
	  if ((*_profile)) delete (*_profile);
	  _Profile.erase(_profile); 
	  return 1;
	}
    }
  return 0;
}
// ________________________________________________________________________

TProfile* THisto::GetProfile(char* name)
{
  // To be checked!!! 
  for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if (!strcmp((*_profile)->GetName(),name)) {return (*_profile);}
    }
  return 0;
}
// ________________________________________________________________________

Int_t THisto::FillProfile(char* name, Float_t valuex, Float_t valuey)
{
 for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if (!strcmp((*_profile)->GetName(),name)) 
	{ (*_profile)->Fill(valuex, valuey); return 1; }
    }
 cout << "Tried to fill nonexisting TProfile " << name << "!\n";
 return 0;
}
// ________________________________________________________________________

Int_t THisto::FillProfile(char* name, Float_t valuex, Float_t valuey, Float_t weightz)
{
 for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if (!strcmp((*_profile)->GetName(),name)) 
	{ (*_profile)->Fill(valuex, valuey, weightz); return 1; }
    }
 cout << "Tried to fill nonexisting TProfile " << name << "!\n";
  return 0;
}
// ________________________________________________________________________

TProfile* THisto::GetFirstProfile()
{
  return (_Profile.front());
}
// ________________________________________________________________________

TProfile* THisto::GetLastProfile()
{
 return (_Profile.back());
}
// ________________________________________________________________________

int THisto::Sumw2(char* name)
{
  for(list<TH1F*>::iterator _h1 = _H1.begin(); _h1!=_H1.end(); _h1++)
    {
      if (!strcmp((*_h1)->GetName(),name)) 
	{ (*_h1)->Sumw2(); return 1; }
    }
  //  return 0;

for(list<TH2F*>::iterator _h2 = _H2.begin(); _h2!=_H2.end(); _h2++)
    {
      if (!strcmp((*_h2)->GetName(),name)) 
	{ (*_h2)->Sumw2(); return 1; }
    }

 for(list<TProfile*>::iterator _profile = _Profile.begin(); _profile!=_Profile.end(); _profile++)
    {
      if (!strcmp((*_profile)->GetName(),name)) 
	{ (*_profile)->Sumw2(); return 1; }
    }
 return 0;
}
//{---------------------------------------------------------------------}
