#include "TestDigit1.h"
#include <TFile.h>
#include <TTree.h>
#include <TRandom.h>
#include <TClonesArray.h>
#include <TObjArray.h>

ClassImp(NaDigit1D)

//____________________________________________
NaDigit1D::NaDigit1D()
{ 
  fI = -1;
  fCluster = -1; 
  fTracks = 0; 
#ifdef STOREFULLINFO
  SetXYZE(0,0,0,0);
#endif
  //
}

//____________________________________________
NaDigit1D::NaDigit1D(Int_t val)
{ 
  fI = val;
  fCluster = -1;
  fTracks = 0; 
#ifdef STOREFULLINFO
  SetXYZE(0,0,0,0);
#endif
  //
}

//____________________________________________
NaDigit1D::~NaDigit1D()
{
  if (fTracks) delete fTracks;
  //
}

//____________________________________________
void NaDigit1D::Print(Option_t *opt) const
{
  printf("Row# %3d |  Digit#%2d of Sensor %d | In Clus# %2d, Track Ptr: %p |",fI,GetIndexInSensor(),GetSensorElemID(),fCluster,fTracks);
#ifdef STOREFULLINFO
  printf(" X: %4.1f Y: %4.1f Z: %4.1f E: %4.1f",fXYZE[0],fXYZE[1],fXYZE[2],fXYZE[3]);
#endif
  printf("\n");
}

//-------------------------------------------------------------------------------------------


ClassImp(NaPixDigit)
//________________________________________________________________
NaPixDigit::NaPixDigit(Int_t col, Int_t row)
  : NaDigit1D(row), fCol(col)
{ 
}
//____________________________________________
void NaPixDigit::Print(Option_t *opt) const
{
  printf("Col# %2d ",fCol);
  NaDigit1D::Print(opt);
}



//////////////////////////////////////////////////////////////////////////////////////////////

Int_t modeW = 1;
Int_t kSplitLev = 0;
Int_t kNTestEv=2;
Int_t NoAuto = 0;
//
int nbyte;
const  Int_t kNSensors=2; // number of for sensors (TClonesArrays with digits) 
TObjArray *Detector=0; // set of sensors
TFile* OutFile;
TFile* InpFile;
TTree* OutTree;
TTree* InpTree;
TBranch* BIn;
TBranch* BOut;
TBranch* BrOut[kNSensors];
TBranch* BrIn[kNSensors];
TClonesArray* sensor[kNSensors];
//

//_____________________________________________________________________
void InitDetector()
{
  if (Detector) return;
  char name[50];
  Detector = new TObjArray(kNSensors);
  for (int i=0;i<kNSensors;i++) {
    sensor[i] = new TClonesArray("NaPixDigit",10);
    sprintf(name,"sens%d",i);
    sensor[i]->SetName(name);
    Detector->Add(sensor[i]);
  }  
  Detector->SetOwner(kTRUE);
}

//_____________________________________________________________________
void SetRead(const char* fname)
{
  InpFile = new TFile(fname);
  InpTree = (TTree*)InpFile->Get("digTree");
  if ((BIn=InpTree->GetBranch("detBranch"))) { // old
    printf("Read OLD format: Single Branch/Detector\n");
    InpTree->SetBranchAddress("detBranch",&Detector);
    //    BIn->SetAutoDelete(kFALSE);
  }
  else {
    printf("Read NEW format: Detector is split in 1 branch/sensor\n");
    for (int i=0;i<kNSensors;i++) {
      BrIn[i] = InpTree->GetBranch(sensor[i]->GetName());
      BrIn[i]->SetAddress(&sensor[i]);
      if (NoAuto) BrIn[i]->SetAutoDelete(kFALSE);
    }    
  }  
}

//_____________________________________________________________________
void Clear() {
  for (int i=0;i<kNSensors;i++) {
    Detector->At(i)->Clear();
  }
}

//_____________________________________________________________________
void SetWrite(const char* fname)
{
  OutFile = new TFile(fname,"RECREATE");
  OutTree = new TTree("digTree","digTree");
  if (modeW==0) {
    printf("Write in OLD format:  Single Branch/Detector\n");
    BOut = OutTree->Branch("detBranch","TObjArray",&Detector,32000,kSplitLev);
  }
  else {
    printf("Write in NEW format: Detector is split in 1 branch/sensor\n");
    //OutTree->Branch(Detector,32000,1);
    for (int i=0;i<kNSensors;i++) {
      BrOut[i] = OutTree->Branch(sensor[i]->GetName(),"TClonesArray",&(sensor[i]),32000,kSplitLev);
      //BrOut[i] = OutTree->GetBranch(sensor[i]->GetName());
    }
  }
}

//_____________________________________________________________________
void ReadEvent(Int_t iev) {
  Clear();
  printf("Event %d\n",iev);    
  nbyte = InpTree->GetEntry(iev);
  printf("Read %d bytes\n",nbyte);
  // old format reading regenerates arrays, redefine
  if (BIn) for (int i=0;i<kNSensors;i++) sensor[i] = (TClonesArray*)Detector->At(i);
  for (int i=0;i<kNSensors;i++) sensor[i]->Print();
}

//_____________________________________________________________________
void WriteEvent() {
  nbyte = OutTree->Fill(); // save event  
  printf("Wrote %d bytes\n",nbyte);
}

//_____________________________________________________________________
void PrintStatus()
{
  printf("\nInput: -------------------------------------------\n");
  if (BIn) printf("Old Detector branch: %s, Address: %p  AutoDelete: %d\n",BIn->GetName(),BIn->GetAddress(),BIn->IsAutoDelete());
  else if (BrIn[0]) {
    for (int i=0;i<kNSensors;i++) printf("New Sensor branch: %s, Address: %p  AutoDelete: %d\n",BrIn[i]->GetName(),BrIn[i]->GetAddress(),BrIn[i]->IsAutoDelete());
  }
  else {printf("N/A\n");}
  //
  printf("\nOutput: -------------------------------------------\n");
  if (BOut) printf("Old Detector branch: %s, Address: %p  AutoDelete: %d\n",BOut->GetName(),BOut->GetAddress(),BOut->IsAutoDelete());
  else if (BrOut[0]) {
    for (int i=0;i<kNSensors;i++) printf("New Sensor branch: %s, Address: %p  AutoDelete: %d\n",BrOut[i]->GetName(),BrOut[i]->GetAddress(),BrOut[i]->IsAutoDelete());
  }
  else {printf("N/A\n");}
  //
  printf("\nState of sensor arrays: \n");
  for (int i=0;i<kNSensors;i++) printf("Name: %s, Size: %d  Last: %d\n",sensor[i]->GetName(),sensor[i]->GetSize(),sensor[i]->GetLast());
  //
}

//_____________________________________________________________________
void TestGen(int writemode=1,const char* fname = "test0.root")
{
  modeW = writemode;
  InitDetector();
  SetWrite(fname);
  //
  // generate test events
  for (int iev=0;iev<kNTestEv;iev++) {
    Clear();
    printf("Event %d\n",iev);
    for (int isn=0;isn<kNSensors;isn++) {
      int ndig = kNTestEv - iev;//gRandom->Integer(10);
      TClonesArray* sensor = (TClonesArray*)Detector->At(isn);
      sensor->Delete(); // clean old digits
      for (int id=0;id<ndig;id++) {
	NaPixDigit* dig = new((*sensor)[id]) NaPixDigit(iev*100+isn*10+id,gRandom->Integer(32));
	dig->SetIndexAndSensorID(id,isn);
	dig->Print();
      }
    }
    OutTree->Fill(); // save event
  }
  //
  PrintStatus();
  //
  OutTree->Write();
  delete OutTree;
  OutFile->Close();
  delete OutFile;
  //
}

//_____________________________________________________________________
void TestRead(const char* fname = "test0.root")
{
  InitDetector();
  //  
  SetRead(fname);
  //
  // read test events
  int nev = (int)InpTree->GetEntries();
  for (int iev=0;iev<nev;iev++) {
    ReadEvent(iev);
  }
  //
  PrintStatus();
  //
  delete InpTree;
  InpFile->Close();
  delete InpFile;
  //
}

//_____________________________________________________________________
void TestReadWrite(int writemode=1,const char* fnameIn = "test0.root",const char* fnameOut = "re_test0.root")
{
  modeW = writemode;
  InitDetector();
  SetWrite(fnameOut);
  SetRead(fnameIn);
  //
  // read test events and write them into new fle
  int nev = (int)InpTree->GetEntries();
  for (int iev=0;iev<nev;iev++) {
    ReadEvent(iev);
    WriteEvent();
  }
  //
  PrintStatus();
  //
  delete InpTree;
  InpFile->Close();
  delete InpFile;

  OutFile->cd();
  OutTree->Write();
  delete OutTree;
  OutFile->Close();
  delete OutFile;

  //
}

