# Authors:
# * Hinnerk C. Schmidt 02/2021
# * Jonas Rembser 03/2021
# * Harshal Shende 06/2021

################################################################################
# Copyright (C) 1995-2020, Rene Brun and Fons Rademakers.                      #
# All rights reserved.                                                         #
#                                                                              #
# For the licensing terms see $ROOTSYS/LICENSE.                                #
# For the list of contributors see $ROOTSYS/README/CREDITS.                    #
################################################################################


from ._utils import _kwargs_to_roocmdargs, cpp_signature


class RooAbsData(object):
    r"""Some member functions of RooAbsData that take a RooCmdArg as argument also support keyword arguments.
    This applies to RooAbsData::plotOn, RooAbsData::createHistogram, RooAbsData::reduce, RooAbsData::statOn.
    For example, the following code is equivalent in PyROOT:
    \code{.py}
    # Directly passing a RooCmdArg:
    data.plotOn(frame, ROOT.RooFit.CutRange("r1"))

    # With keyword arguments:
    data.plotOn(frame, CutRange="r1")
    \endcode
    """
    
    __cpp_name__ = 'RooAbsData'

    @cpp_signature(
        "RooPlot *RooAbsData::plotOn(RooPlot* frame,"
        "			  const RooCmdArg& arg1={}, const RooCmdArg& arg2={},"
        "			  const RooCmdArg& arg3={}, const RooCmdArg& arg4={},"
        "			  const RooCmdArg& arg5={}, const RooCmdArg& arg6={},"
        "			  const RooCmdArg& arg7={}, const RooCmdArg& arg8={}) const ;"
    )
    def plotOn(self, *args, **kwargs):
        r"""The RooAbsData::plotOn() function is pythonized with the command argument pythonization.
        The keywords must correspond to the CmdArgs of the function.
        """
        # Redefinition of `RooAbsData.plotOn` for keyword arguments.
        args, kwargs = _kwargs_to_roocmdargs(*args, **kwargs)
        return self._plotOn(*args, **kwargs)

    @cpp_signature(
        "TH1 *RooAbsData::createHistogram(const char *name, const RooAbsRealLValue& xvar,"
        "                       const RooCmdArg& arg1={}, const RooCmdArg& arg2={},"
        "                       const RooCmdArg& arg3={}, const RooCmdArg& arg4={},"
        "                       const RooCmdArg& arg5={}, const RooCmdArg& arg6={},"
        "                       const RooCmdArg& arg7={}, const RooCmdArg& arg8={}) const ;"
    )
    def createHistogram(self, *args, **kwargs):
        r"""The RooAbsData::createHistogram() function is pythonized with the command argument pythonization.
        The keywords must correspond to the CmdArgs of the function.
        """
        # Redefinition of `RooAbsData.createHistogram` for keyword arguments.
        args, kwargs = _kwargs_to_roocmdargs(*args, **kwargs)
        return self._createHistogram(*args, **kwargs)

    @cpp_signature(
        "RooAbsData *RooAbsData::reduce(const RooCmdArg& arg1,const RooCmdArg& arg2={},"
        "                   const RooCmdArg& arg3={},const RooCmdArg& arg4={},"
        "                   const RooCmdArg& arg5={},const RooCmdArg& arg6={},"
        "                   const RooCmdArg& arg7={},const RooCmdArg& arg8={}) ;"
    )
    def reduce(self, *args, **kwargs):
        r"""The RooAbsData::reduce() function is pythonized with the command argument pythonization.
        The keywords must correspond to the CmdArgs of the function.
        """
        # Redefinition of `RooAbsData.reduce` for keyword arguments.
        args, kwargs = _kwargs_to_roocmdargs(*args, **kwargs)
        return self._reduce(*args, **kwargs)

    @cpp_signature(
        "RooPlot *RooAbsData::statOn(RooPlot* frame,"
        "                          const RooCmdArg& arg1={}, const RooCmdArg& arg2={},"
        "                          const RooCmdArg& arg3={}, const RooCmdArg& arg4={},"
        "                          const RooCmdArg& arg5={}, const RooCmdArg& arg6={},"
        "                          const RooCmdArg& arg7={}, const RooCmdArg& arg8={}) ;"
    )
    def statOn(self, *args, **kwargs):
        r"""The RooAbsData::statOn() function is pythonized with the command argument pythonization.
        The keywords must correspond to the CmdArgs of the function.
        """
        # Redefinition of `RooAbsData.statOn` for keyword arguments.
        args, kwargs = _kwargs_to_roocmdargs(*args, **kwargs)
        return self._statOn(*args, **kwargs)
